/*****************************************************************************
 * main.c
 *
 * Project: SCT Application Example for LPC18xx
 *
 * Description:
 *   Implements the frequency measurement example from the SCT FSM 
 *   designer user manual
 *
 *----------------------------------------------------------------------------
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
 *****************************************************************************/

/* For this application the following interface pins are used:
 * ----------------------------------------------------------------------------------
 * 1) NXP validation board:
 *
 *
 * Signal    Port   Connector           Signal
 * -----------------------------------------------------------------------------------  
 * CTIN_3    P7_3   LCD_UNI, pin 22     IN
 * CTOUT_0   P4_2   LCD_UNI, pin 9      DONE
 *
 * -----------------------------------------------------------------------------------
 *
 *
 * 2) NXP evaluation board v4.1:
 *
 * Signal    Port   Connector           Signal
 * ----------------------------------------------------------------------------------- 
 * CTIN_3    P7_3   X1, pin 22          IN
 * CTOUT_0   P4_2   X1, pin 3           DONE
 *            
 * -----------------------------------------------------------------------------------
 *
 *
 * 3) Hitex board:  
 *
 * Signal    Port   Connector           Signal
 * -----------------------------------------------------------------------------------   
 * CTIN_3    P7_3   SV18, pin 16        IN
 * CTOUT_0   P4_2   X11, pin 7          DONE
 *          
 * -----------------------------------------------------------------------------------
 */
                  
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#ifdef LPC43XX
    #include "LPC43xx_TC2.h"
    #include "scu_LPC43xx.h"
    #include "emc_LPC43xx.h"
#endif

#ifdef LPC18XX
    #include "LPC18xx.h"
    #include "scu_LPC18xx.h"
    #include "emc_LPC18xx.h"
#endif


#include "sct_fsm.h"
#include "timer.h"
#include "uarthandler.h"
#include "type.h"


#define NUM_CAPTURE 16

uint32_t sample_buffer[NUM_CAPTURE];
uint32_t freq_rising[NUM_CAPTURE/2];
uint32_t freq_falling[NUM_CAPTURE/2];

volatile int wait = 1;



// Macros for 2 buttons on the LPC43xx/LPC18xx evaluation board from NXP
// SW2 and SW3 are connected to GPIOs
// SW4 and SW5 are intended to work as ADCTRIG[1:0] inputs
#ifdef NXP_EVAL_BOARD
    #define SW2 !((LPC_GPIO3->PIN>>6)&1)    // P6.10 - GPIO3[6]
    #define SW3 !((LPC_GPIO2->PIN>>0)&1)    // P4.0 - GPIO2[0]
#endif


  
/******************************************************************************
 * Initialize board type specific IO's:
 *  - NXP validation board
 *  - NXP evaluation board v4.1
 *  - Hitex board version A2
 ******************************************************************************/
void IOInit(void)
{

#ifdef NXP_VALIDATION_BOARD 
    // Board (and chip) specific initializations - NXP validation board 
    #ifdef LPC43XX 
        scu_pinmux(9, 2, MD_PDN, FUNC2);    // GPIO4_14: LD11   
        scu_pinmux(0xA ,4 , MD_PDN, FUNC2); // GPIO4_11: LD10
        LPC_GPIO4->DIR |= (1<<14);          // GPIO4_11 = output
        LPC_GPIO4->DIR |= (1<<11);          // GPIO4_11 = output

        scu_pinmux(6 ,10, MD_PUP, FUNC3);   // GPIO3_6: SW4 (button 0)
        scu_pinmux(4 , 0, MD_PUP, FUNC0);   // GPIO2_0: SW3 (button 1)
        LPC_GPIO3->DIR &= ~(1<<6);          // Button 0 = input
        LPC_GPIO2->DIR &= ~(1<<0);          // Button 1 = input
    
        // Pin configuration for SCT inputs/outputs - NXP validation board 
        scu_pinmux (4, 2, MD_PUP, FUNC1);   // P4_2:  SCTOUT_0 - LCD_UNI, pin 9 
        scu_pinmux (4, 1, MD_PLN, FUNC1);   // P4_1:  SCTOUT_1 - LCD_UNI, pin 6
        scu_pinmux (4, 4, MD_PLN, FUNC1);   // P4_4:  SCTOUT_2 - LCD_UNI, pin 7
        scu_pinmux (4, 3, MD_PLN, FUNC1);   // P4_3:  SCTOUT_3 - LCD_UNI, pin 8
        scu_pinmux (4, 6, MD_PLN, FUNC1);   // P4_6:  SCTOUT_4 - LCD_UNI, pin 35
        scu_pinmux (4, 5, MD_PLN, FUNC1);   // P4_5:  SCTOUT_5 - LCD_UNI, pin 28
    
        scu_pinmux (2, 3, MD_PDN, FUNC0);   // P2_3:  SCTIN_1 - I2C_1, pin 7
        // SCTIN_2 does not work on the LPC4300 test chip
        //scu_pinmux (4,10, MD_PDN, FUNC1);   // P4_10: SCTIN_2 - LCD_UNI, pin 16 
        scu_pinmux (7, 3, MD_PDN, FUNC1);   // P7_3:  SCTIN_3 - LCD_UNI, pin 22 
    #endif

    #ifdef LPC18XX 
        scu_pinmux(9, 2, MD_PDN, FUNC0);    // GPIO4_14: LD11

        // on LPC1800 port pin PA_4 is not available    
        //scu_pinmux(0xA ,4 , MD_PDN, FUNC2); // GPIO4_11: LD10
        //LPC_GPIO4->DIR |= (1<<14);            // GPIO4_11 = output

        scu_pinmux(6 ,10, MD_PUP, FUNC3);   // GPIO3_6: SW4 (button 0)
        scu_pinmux(4 , 0, MD_PUP, FUNC0);   // GPIO2_0: SW3 (button 1)
    
        LPC_GPIO4->DIR |= (1<<14);          // LD11 = output
        LPC_GPIO3->DIR &= ~(1<<6);          // Button 0 = input
        LPC_GPIO2->DIR &= ~(1<<0);          // Button 1 = input
    
        // Pin configuration for SCT inputs/outputs - NXP validation board 
        scu_pinmux (4, 2, MD_PUP, FUNC1);   // P4_2:  SCTOUT_0 - LCD_UNI, pin 9 
        scu_pinmux (4, 1, MD_PLN, FUNC1);   // P4_1:  SCTOUT_1 - LCD_UNI, pin 6
        scu_pinmux (4, 4, MD_PLN, FUNC1);   // P4_4:  SCTOUT_2 - LCD_UNI, pin 7
        scu_pinmux (4, 3, MD_PLN, FUNC1);   // P4_3:  SCTOUT_3 - LCD_UNI, pin 8
        scu_pinmux (4, 6, MD_PLN, FUNC1);   // P4_6:  SCTOUT_4 - LCD_UNI, pin 35
        scu_pinmux (4, 5, MD_PLN, FUNC1);   // P4_5:  SCTOUT_5 - LCD_UNI, pin 28
    
        scu_pinmux (2, 3, MD_PDN, FUNC3);   // P2_3:  SCTIN_1 - I2C_1, pin 7
        scu_pinmux (4,10, MD_PDN, FUNC1);   // P4_10: SCTIN_2 - LCD_UNI, pin 16 
        scu_pinmux (7, 3, MD_PDN, FUNC1);   // P7_3:  SCTIN_3 - LCD_UNI, pin 22 
    #endif
#endif


#ifdef NXP_EVAL_BOARD   
    // Board (and chip) specific initializations - NXP eval board v4.1
    #ifdef LPC43XX
        scu_pinmux(8, 0, MD_PLN, FUNC3);    // GPIO4_0: on-board LED0   
        scu_pinmux(8 ,1, MD_PLN, FUNC3);    // GPIO4_1: on-board LED1
        scu_pinmux(6,10, MD_PUP, FUNC3);    // GPIO3_6: on board SW2 (button 0)
        scu_pinmux(4, 0, MD_PUP, FUNC0);    // GPIO2_0: on-board SW3 (button 1)
    
        LPC_GPIO4->DIR |= 0x0003;           // LED0 and LED1 = output
        LPC_GPIO3->DIR &= ~(1<<6);          // Button 0 = input
        LPC_GPIO2->DIR &= ~(1<<0);          // Button 1 = input
    
        // Pin configuration for SCT inputs/outputs - NXP eval board v4.1
        scu_pinmux (4, 2, MD_PUP, FUNC1);   // P4_2:  SCTOUT_0 - X1, pin 3
        scu_pinmux (4, 1, MD_PLN, FUNC1);   // P4_1:  SCTOUT_1 - X1, pin 4
        scu_pinmux (4, 4, MD_PLN, FUNC1);   // P4_4:  SCTOUT_2 - X1, pin 5
        scu_pinmux (4, 3, MD_PLN, FUNC1);   // P4_3:  SCTOUT_3 - X1, pin 6
        scu_pinmux (4, 6, MD_PLN, FUNC1);   // P4_6:  SCTOUT_4 - X1, pin 7
        scu_pinmux (4, 5, MD_PLN, FUNC1);   // P4_5:  SCTOUT_5 - X1, pin 8
    
        scu_pinmux (0xD, 10, MD_PDN, FUNC1);    // PD_10: SCTIN_1 - X1, pin 20
        // SCTIN_2 does not work on the LPC4300 test chip
        //scu_pinmux (2, 5,    MD_PDN, FUNC1);      // P2_5:  SCTIN_2 - JP7, pin 1
        scu_pinmux (7, 3,    MD_PDN, FUNC1);    // P7_3:  SCTIN_3 - X1, pin 22
    #endif

    #ifdef LPC18XX
        scu_pinmux(8, 0, MD_PLN, FUNC0);    // GPIO4_0: on-board LED0   
        scu_pinmux(8 ,1, MD_PLN, FUNC0);    // GPIO4_1: on-board LED1
        scu_pinmux(6,10, MD_PUP, FUNC0);    // GPIO3_6: on board SW2 (button 0)
        scu_pinmux(4, 0, MD_PUP, FUNC0);    // GPIO2_0: on-board SW3 (button 1)
    
        LPC_GPIO4->DIR |= 0x0003;           // LED0 and LED1 = output
        LPC_GPIO3->DIR &= ~(1<<6);          // Button 0 = input
        LPC_GPIO2->DIR &= ~(1<<0);          // Button 1 = input
    
        // Pin configuration for SCT inputs/outputs - NXP eval board v4.1
        scu_pinmux (4, 2, MD_PUP, FUNC1);   // P4_2:  SCTOUT_0 - X1, pin 3
        scu_pinmux (4, 1, MD_PLN, FUNC1);   // P4_1:  SCTOUT_1 - X1, pin 4
        scu_pinmux (4, 4, MD_PLN, FUNC1);   // P4_4:  SCTOUT_2 - X1, pin 5
        scu_pinmux (4, 3, MD_PLN, FUNC1);   // P4_3:  SCTOUT_3 - X1, pin 6
        scu_pinmux (4, 6, MD_PLN, FUNC1);   // P4_6:  SCTOUT_4 - X1, pin 7
        scu_pinmux (4, 5, MD_PLN, FUNC1);   // P4_5:  SCTOUT_5 - X1, pin 8
    
        scu_pinmux (0xD, 10, MD_PDN, FUNC1);    // PD_10: SCTIN_1 - X1, pin 20
        scu_pinmux (2, 5,    MD_PDN, FUNC1);    // P2_5:  SCTIN_2 - JP7, pin 1
        scu_pinmux (7, 3,    MD_PDN, FUNC1);    // P7_3:  SCTIN_3 - X1, pin 22
    #endif
#endif

#ifdef HITEX_BOARD
    // Board (and chip) specific initializations - Hitex board
    #ifdef LPC43XX
        scu_pinmux(1 ,15 , MD_PLN, FUNC2);  // GPIO0_2: SV6, pin 11 used for external LED0
        scu_pinmux(1 ,16 , MD_PLN, FUNC2);  // GPIO0_3: SV6, pin 5 used for external LED1
        scu_pinmux(1 ,18 , MD_PUP, FUNC3);  // GPIO0_13: SV3, pin 11 used for external button 0
        scu_pinmux(1 ,20 , MD_PUP, FUNC3);  // GPIO0_15: SV3, pin 9 used for external button 1

        LPC_GPIO0->DIR |= 0x000A;           // LED0 and LED1 = output
        LPC_GPIO0->DIR &= ~(0xA000);        // Button 0 and Button 1 = input
        
        // Pin configuration for SCT - Hitex board
        scu_pinmux(4, 2, MD_PUP, FUNC1);   // P4_2:  SCTOUT_0 - X11, pin 7
        scu_pinmux(4, 1, MD_PLN, FUNC1);   // P4_1:  SCTOUT_1 - JP20, pin 2
        scu_pinmux(4, 4, MD_PLN, FUNC1);   // P4_4:  SCTOUT_2 - JP22, pin 2
        scu_pinmux(4, 3, MD_PLN, FUNC1);   // P4_3:  SCTOUT_3 - X11, pin 6
        scu_pinmux(4, 6, MD_PLN, FUNC1);   // P4_6:  SCTOUT_4 - X11, pin 35
        scu_pinmux(4, 5, MD_PLN, FUNC1);   // P4_5:  SCTOUT_5 - X11, pin 28
        scu_pinmux(7, 7, MD_PLN, FUNC1);   // P7_7:  SCTOUT_8 - X11, pin 34
        scu_pinmux(0xB, 0, MD_PLN, FUNC1); // PB_0:  SCTOUT_10 - X11, pin 27
        scu_pinmux(7, 6, MD_PLN, FUNC1);   // P7_6:  SCTOUT_11 - X11, pin 30
        scu_pinmux(7, 5, MD_PLN, FUNC1);   // P7_5:  SCTOUT_12 - SV18, pin 2
        scu_pinmux(7, 0, MD_PLN, FUNC1);   // P7_0:  SCTOUT_14 - JP19, pin 1
        scu_pinmux(7, 1, MD_PLN, FUNC1);   // P7_1:  SCTOUT_15 - X11, pin 22
    
        scu_pinmux(2, 4,  MD_PUP, FUNC0);  // P2_4:  SCTIN_0 - X3, pin 14
        scu_pinmux(2, 3,  MD_PUP, FUNC0);  // P2_3:  SCTIN_1 - X3, pin 11
        // SCTIN_2 does not work on the LPC4300 test chip
        //scu_pinmux(4, 10, MD_PUP, FUNC1);  // P4_10: SCTIN_2 - X11, pin 13    // does not work
        //scu_pinmux(0xF, 8, MD_PUP, FUNC0); // PF_8: SCTIN_2 - X5, pin 20      // does not work
        //scu_pinmux(2, 5,  MD_PUP, FUNC1);  // P2_5: SCTIN_2 - IC U13, pin 8   // does not work
        scu_pinmux(7, 3,  MD_PUP, FUNC1);  // P7_3:  SCTIN_3 - SV18, pin 16
        scu_pinmux(7, 2,  MD_PUP, FUNC1);  // P7_2:  SCTIN_4 - X11, pin 21
        scu_pinmux(4, 9,  MD_PUP, FUNC1);  // P4_9:  SCTIN_6 - X11, pin 11
    #endif
    
    #ifdef LPC18XX
        scu_pinmux(1 ,15 , MD_PLN, FUNC0);  // GPIO0_2: SV6, pin 11 used for external LED0
        scu_pinmux(1 ,16 , MD_PLN, FUNC0);  // GPIO0_3: SV6, pin 5 used for external LED1
        scu_pinmux(1 ,18 , MD_PUP, FUNC0);  // GPIO0_13: SV3, pin 11 used for external button 0
        scu_pinmux(1 ,20 , MD_PUP, FUNC0);  // GPIO0_15: SV3, pin 9 used for external button 1

        LPC_GPIO0->DIR |= 0x000A;           // LED0 and LED1 = output
        LPC_GPIO0->DIR &= ~(0xA000);        // Button 0 and Button 1 = input
        
        // Pin configuration for SCT - Hitex board
        scu_pinmux (4, 2, MD_PUP, FUNC1);   // P4_2:  SCTOUT_0 - X11, pin 7
        scu_pinmux (4, 1, MD_PLN, FUNC1);   // P4_1:  SCTOUT_1 - JP20, pin 2
        scu_pinmux (4, 4, MD_PLN, FUNC1);   // P4_4:  SCTOUT_2 - JP22, pin 2
        scu_pinmux (4, 3, MD_PLN, FUNC1);   // P4_3:  SCTOUT_3 - X11, pin 6
        scu_pinmux (4, 6, MD_PLN, FUNC1);   // P4_6:  SCTOUT_4 - X11, pin 35
        scu_pinmux (4, 5, MD_PLN, FUNC1);   // P4_5:  SCTOUT_5 - X11, pin 28
    
        scu_pinmux(2, 4,  MD_PUP, FUNC3);  // P2_4:  SCTIN_0 - X3, pin 14
        scu_pinmux(2, 3,  MD_PUP, FUNC3);  // P2_3:  SCTIN_1 - X3, pin 11
        scu_pinmux(4, 10, MD_PUP, FUNC1);  // P4_10: SCTIN_2 - X11, pin 13
        scu_pinmux(7, 3,  MD_PUP, FUNC1);  // P7_3:  SCTIN_3 - SV18, pin 16
        scu_pinmux(7, 2,  MD_PUP, FUNC1);  // P7_2:  SCTIN_4 - X11, pin 21
        scu_pinmux(4, 9,  MD_PUP, FUNC1);  // P4_9:  SCTIN_6 - X11, pin 11      
    #endif
#endif
}



/****************************************************************************************
*  timer_delayMs
*
*   
****************************************************************************************/
void timer_delayMs(uint32_t delayInMs)
{

    LPC_TIMER0->TCR = 0x02;     // reset timer 
    LPC_TIMER0->PR  = 0x00;     // set prescaler to zero

    #ifdef LPC43XX  
        LPC_TIMER0->MR[0] = delayInMs * ((M4Frequency / 1000) - 1);
    #endif

    #ifdef LPC18XX  
        LPC_TIMER0->MR[0] = delayInMs * ((M3Frequency / 1000) - 1);
    #endif

    LPC_TIMER0->IR  = 0xFF;     // reset all interrrupts 
    LPC_TIMER0->MCR = 0x04;     // stop timer on match
    LPC_TIMER0->TCR = 0x01;     // start timer
    
    // Start timer and wait until delay time has elapsed 
    while (LPC_TIMER0->TCR & 0x01);

}

 


/****************************************************************************************
*  SCT_IRQHandler
*
*   
****************************************************************************************/
void SCT_IRQHandler (void)
{
    /* Acknowledge the interrupt source */
    LPC_SCT->EVFLAG = (1 << SCT_IRQ_EVENT_samplingComplete);

    /* Let 'main' know that sampling has finished */
    wait = 0;
}



/****************************************************************************************
*
*
*   
****************************************************************************************/
int main (void) 
{
    int i;
    char c[100];

    #ifdef NXP_EVAL_BOARD
        static char status;
    #endif

    #ifdef EXTFLASH
        extern unsigned long __Vectors; 
        SCB->VTOR = (unsigned long)&__Vectors;
    #endif


    // System and board specific initializations 
    SystemInit();

    // Enable clocks and PLL
    LPC_CREG->CREG0 |= (1<<1)|(1<<0);               // Enable 32 kHz & 1 kHz on osc32k 
    SetPL160M(SRC_IRC, 10);                         // Set PL160M @ 10*12 = 120 MHz with the IRC as source clock
    #ifdef LPC43XX 
        SetClock(BASE_M4_CLK, SRC_PL160M_0, DIV1);      // Run base M4 clock from PL160M, div by 1 = no division 
    #endif
    #ifdef LPC18XX 
        SetClock(BASE_M3_CLK, SRC_PL160M_0, DIV1);      // Run base M3 clock from PL160M, div by 1 = no division 
    #endif

    // Initialise the IO's needed for this example
    IOInit();

    #ifdef EXTFLASH
        EMC_Config_Pinmux();
        EMC_Config_Static();
    #endif
    
    // Initialise UART1 as output channel 
    SetClock(BASE_UART1_CLK, SRC_PL160M_0, DIV1);
    // The setting for 115200@120MHz processor frequency is hardcoded in UART1Init.
    // Either change the settings in UART1Init or remove the compiler define "DLL_DLM"
    // then the correct settings for the fractional divider are calculated at run time.
    UART1Init(115200);


    // Global configuration of the SCT 
    #ifdef LPC18XX
        LPC_CCU1->CLK_M3_SCT_CFG = 0x0001;          /* Enable SCT branch clock in CCU1, RUN=1, AUTO=0, WAKE=0 */
    #endif
    #ifdef LPC43XX
        LPC_CCU1->CLK_M4_SCT_CFG = 0x0001;          /* Enable SCT branch clock in CCU1, RUN=1, AUTO=0, WAKE=0 */
    #endif

    LPC_SCT->CONFIG = 0x00000000;               /* Configure as two separate 16-bit timers, clocked internally */
  
    LPC_SCT->CTRL_L = 0x000C + ((1-1) << 5);    /* L counter: */
                                                /* Set prescaler = 1. Clear counter. Keep halted */
                                              
    //LPC_SCT->CTRL_H = 0x000C + ((1-1) << 5);  /* H counter: */
                                                /* Set prescaler = 1. Clear counter. Keep halted */

    /* Now use the FSM code to configure the state machine */
    sct_fsm_init();

    NVIC_EnableIRQ (SCT_IRQn);  
    
    // Run in this loop forever and make measurements every x seconds
    while( 1 )
    {
        // Start the SCT
        LPC_SCT->CTRL_U &= ~(1 << 2);  /* Run L only */
    
        // run cpu into a loop waiting for completion of the sampling
        while (wait);
        
         
        // Now put the results from the SCT into the buffer 
        sample_buffer[0] = SCT_CAPTURE_sample1;
        sample_buffer[1] = SCT_CAPTURE_sample2;
        sample_buffer[2] = SCT_CAPTURE_sample3;
        sample_buffer[3] = SCT_CAPTURE_sample4;
        sample_buffer[4] = SCT_CAPTURE_sample5;
        sample_buffer[5] = SCT_CAPTURE_sample6;
        sample_buffer[6] = SCT_CAPTURE_sample7;
        sample_buffer[7] = SCT_CAPTURE_sample8;
        sample_buffer[8] = SCT_CAPTURE_sample9;
        sample_buffer[9] = SCT_CAPTURE_sample10;
        sample_buffer[10] = SCT_CAPTURE_sample11;
        sample_buffer[11] = SCT_CAPTURE_sample12;
        sample_buffer[12] = SCT_CAPTURE_sample13;
        sample_buffer[13] = SCT_CAPTURE_sample14;
        sample_buffer[14] = SCT_CAPTURE_sample15;
        sample_buffer[15] = SCT_CAPTURE_sample16;
        
        for (i = 0; i < (NUM_CAPTURE/2 - 1); i++)
        {
            // This calculation includes a ceratin error, because of the division with integer values
            freq_rising[i] = PL160M_0Frequency / (sample_buffer[(i*2)+2] - sample_buffer[i*2]);     
        }
        
        for (i = 0; i < (NUM_CAPTURE/2 - 1); i++)
        {
            // This calculation includes a ceratin error, because of the division with integer values
            freq_falling[i] = PL160M_0Frequency / (sample_buffer[(i*2)+3] - sample_buffer[(i*2)+1]);    
        }
        /*
        for (i = 0; i < (NUM_CAPTURE/2 - 1); i++)
        {
            // Make a simple printout of the detected values
            sprintf(c, "fr[%d] = %d   ff[%d] = %d\r\n", i, freq_rising[i], i, freq_falling[i]);
            writeString( c );
        }
        */

        /*
        // Print the sample buffer content
        for (i = 0; i < 16; i++)
        {
            // Make a simple printout of the detected values
            sprintf(c, "sb[%d] = %d\r\n", i, sample_buffer[i]);
            writeString( c );
        }
        */

        // Make a simple printout of the detected values
        sprintf(c, "Frequency = %d Hz\r\n", freq_rising[0]);
        writeString( c );

        wait = 1;

        // Toggle some LED's for whatever reasons
        #ifdef NXP_EVAL_BOARD
            if (status)
            {
                // Turn LED1 - 4 on
                LPC_GPIO4->SET = 0xF0;
                //LPC_GPIO2->SET = 0x02;
                status = FALSE;
            }
            else
            {
                // Turn LED1 - 4 off
                LPC_GPIO4->CLR = 0xF0;
                //LPC_GPIO2->CLR = 0x02;
                status = TRUE;
            }
        #endif

        // Program the timer to wait for a while
        timer_delayMs( 2000 );
   }
        
}


